<?php

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class Widget_TommusRhodus_Hero_Header_Block extends Widget_Base {
	
	//Return Class Name
	public function get_name() {
		return 'tommusrhodus-hero-header-block';
	}
	
	//Return Block Title (for blocks list)
	public function get_title() {
		return esc_html__( 'Hero Header', 'tr-framework' );
	}
	
	//Return Block Icon (for blocks list)
	public function get_icon() {
		return 'eicon-call-to-action';
	}
	
	public function get_categories() {
		return [ 'uptime-elements' ];
	}

	protected function _register_controls() {
		
		$this->start_controls_section(
			'section_layout', [
				'label' => esc_html__( 'Layout', 'tr-framework' ),
			]
		);
		
		$this->add_control(
			'layout', [
				'label'   => __( 'Hero Header', 'tr-framework' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'standard',
				'label_block' => true,
				'options' => [
					'standard'         						=> esc_html__( 'Standard', 'tr-framework' ),
					'standard-alt'         					=> esc_html__( 'Standard, Alternative Decorations', 'tr-framework' ),
					'image-background' 						=> esc_html__( 'Image Background', 'tr-framework' ),
					'dark-background' 						=> esc_html__( 'Dark Background', 'tr-framework' ),
					'dark-background-alert' 				=> esc_html__( 'Dark Background + Alert', 'tr-framework' ),
					'dark-background-image-bottom' 			=> esc_html__( 'Dark Background + Image Bottom', 'tr-framework' ),
					'fullscreen' 							=> esc_html__( 'Fullscreen', 'tr-framework' ),
					'centered-bg-primary-3-background'		=> esc_html__( 'Primary 3 Background, Centered Text', 'tr-framework' )
				],
			]
		);
		
		$this->add_control(
			'divider', [
				'label'       => __( 'Bottom Divider Shape', 'tr-framework' ),
				'type'        => Controls_Manager::SELECT,
				'default'     => 'none',
				'options'     => tommusrhodus_get_divider_layouts(),
				'label_block' => true
			]
		);

		$this->add_control(
			'show_breadcrumbs', [
				'label' 		=> __( 'Show Breadcrumbs Where Applicable?', 'tr-framework' ),
				'type' 			=> Controls_Manager::SWITCHER,
				'label_on' 		=> __( 'Show', 'tr-framework' ),
				'label_off' 	=> __( 'Hide', 'tr-framework' ),
				'return_value' 	=> 'yes',
				'default' 		=> 'yes',
			]
		);
		
		$this->end_controls_section();
		
		$this->start_controls_section(
			'section_content', [
				'label' => esc_html__( 'Content', 'tr-framework' ),
			]
		);
		
		$this->add_control(
			'image', [
				'label'      => __( 'Image', 'tr-framework' ),
				'type'       => Controls_Manager::MEDIA,
				'default' => [
					'url' => '',
				],
			]
		);
		
		$this->add_control(
			'content', [
				'label'       => __( 'Content', 'tr-framework' ),
				'type'        => Controls_Manager::WYSIWYG,
				'default'     => ''
			]
		);

		$this->add_control(
			'alert_content', [
				'label'       => __( 'Alert Content', 'tr-framework' ),
				'type'        => Controls_Manager::WYSIWYG,
				'default'     => ''
			]
		);

		$this->end_controls_section();

	}
	
	/**
	 * Whether the reload preview is required or not.
	 *
	 * Used to determine whether the reload preview is required.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return bool Whether the reload preview is required.
	 */
	public function is_reload_preview_required() {
		return true;
	}

	protected function render() {
		
		$settings = $this->get_settings_for_display();
		
		if( 'standard' == $settings['layout'] ){
		
			echo '
				<section class="bg-primary-alt header-inner o-hidden">
				
					<div class="container">';

						if( 'yes' == $settings['show_breadcrumbs'] || !empty( $settings['show_breadcrumbs'] ) ) {

							echo '
								<div class="row my-3">
									<div class="col">'. get_tommusrhodus_breadcrumbs() .'</div>
								</div>
							';

						}
						
						echo '
						<div class="row py-6 text-center justify-content-center align-items-center">
							<div class="col-xl-8 col-lg-10">'. $settings['content'] .'</div>
						</div>
						
					</div>
				
					<div class="decoration-wrapper d-none d-md-block">
					
						<div class="decoration left middle-y scale-2" data-jarallax-element="100">
							<svg class="bg-primary" width="81" height="99" viewBox="0 0 81 99" fill="none" xmlns="http://www.w3.org/2000/svg">
								<path d="M16.512 39.034C15.694 39.503 14.793 39.858 14.094 40.452C12.276 41.993 10.356 43.475 8.82596 45.278C6.21296 48.36 6.58998 51.078 9.57798 54.153C10.094 54.685 10.955 54.907 11.685 55.194C12.044 55.337 12.476 55.28 12.867 55.352C15.017 55.748 15.705 56.688 15.555 59.035C11.016 60.872 7.36398 59.407 4.22098 55.993C2.15998 53.756 0.741962 51.231 0.661962 48.11C0.632962 46.994 0.790977 45.962 1.37098 44.914C3.54698 40.972 6.52296 37.87 10.412 35.592C11.707 34.833 13.039 34.918 14.351 35.188C15.306 35.385 15.997 36.129 16.206 37.163C16.34 37.815 16.422 38.479 16.512 39.034Z" fill="black" />
								<path d="M15.697 74.223C12.558 75.928 10.693 78.761 9.33598 81.818C7.54498 85.852 9.06098 90.272 14.078 90.917C14.887 91.021 15.717 91.46 16.394 91.946C17.511 92.749 17.462 92.817 17.347 94.582C14.38 95.921 11.482 95.849 8.72196 94.045C7.45596 93.218 6.21997 92.335 5.04597 91.382C1.89797 88.829 0.949985 81.801 2.99699 78.751C4.01799 77.23 4.76297 75.515 5.83497 74.034C7.72997 71.419 10.22 70.101 13.702 71.617C14.943 72.155 15.049 73.297 15.697 74.223Z"
								fill="black" />
								<path d="M78.5449 75.006C76.3729 74.601 75.3419 76.242 74.1429 77.555C72.8619 78.959 72.1569 80.623 72.2269 82.559C72.3869 87.028 74.3889 90.526 78.0839 92.942C79.2399 93.698 79.3539 94.837 80.1699 95.958C79.5739 95.958 79.1369 95.825 78.8579 95.978C76.0829 97.503 74.0079 96.236 72.0439 94.367C70.2509 92.66 68.4229 91.002 67.2859 88.732C66.0829 86.332 65.3249 83.839 65.5929 81.135C66.0289 76.765 68.6159 73.762 71.8999 71.146C73.3389 70 74.8039 70.028 76.3999 70.414C77.5559 70.692 78.1069 71.489 78.2359 72.637C78.3159 73.379 78.4279 74.121 78.5449 75.006Z"
								fill="black" />
								<path d="M50.3418 83.795C47.8708 83.591 45.2398 83.449 42.7048 84.279C41.0388 88.606 42.7048 93.074 46.9178 95.201C48.1448 95.82 49.0408 96.512 49.3608 98.029C48.3408 98.302 47.3058 98.584 46.2688 98.859C45.2998 99.118 44.3958 98.97 43.4818 98.543C37.4038 95.701 33.8938 88.395 35.5428 81.908C35.7188 81.217 36.0308 80.558 36.2988 79.892C36.6208 79.089 37.2028 78.615 38.0698 78.451C41.0778 77.885 44.0858 77.664 47.1268 78.22C48.5698 78.485 49.5718 79.389 50.4468 80.41C51.2928 81.396 51.4728 82.543 50.3418 83.795Z"
								fill="black" />
								<path d="M78.367 7.67896C76.728 7.85196 75.308 8.00099 73.888 8.15199C72.019 8.34999 70.921 9.27497 70.372 11.176C69.458 14.353 69.804 17.353 71.183 20.254C71.49 20.9 72.251 21.386 72.896 21.802C74.689 22.958 76.613 23.942 77.738 26.096C75.463 27.089 73.3 27.21 71.023 26.032C65.492 23.173 63.347 18.18 63.968 12.031C64.157 10.158 64.874 8.31997 65.493 6.51597C66.048 4.90197 67.288 3.77699 68.946 3.48899C70.899 3.14999 72.905 3.07497 74.891 2.98597C75.942 2.93897 76.723 3.57297 77.149 4.50397C77.604 5.49397 77.938 6.53796 78.367 7.67896Z"
								fill="black" />
								<path d="M45.898 11.393C44.492 10.616 44.472 10.618 43.916 11.645C42.529 14.197 42.822 16.754 44.828 18.882C45.478 19.571 46.219 20.213 47.016 20.72C48.481 21.652 49.954 22.528 50.192 24.534C49.585 24.68 48.784 24.712 48.153 25.056C46.801 25.793 45.633 25.606 44.323 24.907C41.643 23.479 39.428 21.542 37.858 18.984C36.092 16.107 35.942 12.919 36.899 9.745C37.524 7.676 39.237 6.54302 41.2 5.90702C42.944 5.34102 44.739 4.91199 46.532 4.51999C48.641 4.05999 50.067 4.95502 50.714 7.04902C50.923 7.72402 51.066 8.42101 51.304 9.36701C50.204 9.76101 49.103 10.138 48.017 10.549C47.292 10.823 46.589 11.147 45.876 11.449L45.898 11.393Z"
								fill="black" />
								<path d="M19.6431 5.52198C18.6041 5.46998 17.909 5.46799 17.219 5.39399C15.08 5.16299 13.449 6.00099 12.369 7.80299C11.473 9.29899 10.721 10.88 9.90007 12.434C10.222 12.967 10.517 13.344 10.697 13.77C11.25 15.085 12.349 15.408 13.611 15.439C14.166 15.452 14.722 15.377 15.279 15.369C17.363 15.334 18.3261 16.139 18.6771 18.216C18.7301 18.522 18.7101 18.839 18.7241 19.13C13.6791 21.07 8.77708 19.623 5.57808 16.217C4.24008 14.791 3.58807 12.961 3.85307 10.912C4.19307 8.30596 5.33908 6.02299 6.79408 3.91099C7.83508 2.39799 9.59309 1.80197 11.2391 1.23597C12.4271 0.826969 13.739 0.693967 15.007 0.614967C17.064 0.485967 18.028 1.20798 18.757 3.16498C19.018 3.86498 19.2811 4.56298 19.6431 5.52198Z"
								fill="black" />
								<path d="M44.8359 43.951C44.5839 44.067 44.4469 44.168 44.2969 44.193C40.0179 44.92 39.9039 48.017 41.3909 51.291C42.1799 53.03 43.8169 54.395 45.1229 55.884C45.5369 56.354 46.0839 56.738 46.6349 57.048C48.2289 57.946 49.4869 59.055 49.4079 61.244C48.3509 61.41 47.1399 61.677 45.9199 61.754C45.4219 61.787 44.8589 61.477 44.3909 61.209C40.5649 59.016 37.5139 55.98 35.4109 52.143C33.4579 48.585 33.1489 44.837 35.5809 41.295C36.9579 39.289 39.1669 38.775 41.4309 38.563C42.2079 38.49 42.7529 38.969 43.1599 39.637C43.9489 40.933 44.6229 42.266 44.8359 43.951Z"
								fill="black" />
								<path d="M76.0799 56.7771C73.8919 57.3361 71.6839 57.078 69.5139 56.634C67.7349 56.269 66.5409 54.89 65.9769 53.335C64.9499 50.503 63.8829 47.5892 63.5389 44.6272C63.0189 40.1442 65.2439 37.073 70.0369 36.672C70.5139 36.632 70.9939 36.615 71.4709 36.621C74.4729 36.654 75.3849 37.559 75.6639 40.847C74.8769 40.847 74.0979 40.84 73.3179 40.848C70.7069 40.875 69.2299 42.4121 69.6129 44.9371C69.9199 46.9721 70.4179 48.985 70.9449 50.978C71.2749 52.235 72.1329 52.87 73.5349 53.043C76.1739 53.373 76.5329 54.0101 76.0799 56.7771Z"
								fill="black" />
							</svg>			
						</div>
						<div class="decoration bottom right scale-2" data-jarallax-element="100 100">
							<svg class="bg-primary-2" width="156" height="159" viewBox="0 0 156 159" fill="none" xmlns="http://www.w3.org/2000/svg">
								<path d="M6.81116 60.239C7.31116 59.53 7.90115 58.866 8.29715 58.102C9.43415 55.902 10.3812 53.598 11.6092 51.453C13.1932 48.69 14.9122 45.997 16.6932 43.356C18.3842 40.847 20.0892 38.319 22.0472 36.02C25.1492 32.374 28.3672 28.812 31.7232 25.398C34.7972 22.272 38.0242 19.28 41.3582 16.433C43.7172 14.419 46.3642 12.738 48.8912 10.924C49.6492 10.38 50.4442 9.88499 51.1212 9.20899C48.9962 9.30999 47.5602 10.816 45.6132 11.487C46.3102 10.867 46.9252 10.1 47.7182 9.65199C53.0772 6.62599 58.4682 3.65297 64.3722 1.78297C64.6632 1.69097 64.9792 1.68298 65.3472 1.78398C61.4882 4.32098 56.8802 5.36497 52.4762 8.12397C53.4332 8.24397 53.8862 8.44497 54.2032 8.31397C55.2362 7.89097 56.2172 7.34298 57.2212 6.84998C61.5302 4.73198 66.0322 3.12597 70.5492 1.51597C75.3952 -0.21203 80.2012 -0.0660178 85.1602 0.659982C90.6312 1.46198 95.9592 2.81399 101.281 4.24699C104.906 5.22299 108.551 6.12599 112.158 7.16099C119.318 9.21199 125.765 12.666 131.603 17.269C133.171 18.506 134.644 19.869 136.234 21.079C139.023 23.201 141.045 25.948 142.882 28.872C143.304 29.543 143.841 30.143 144.345 30.761C146.118 32.939 147.763 35.196 148.761 37.853C150.111 41.439 151.693 44.959 152.738 48.632C154.287 54.083 155.381 59.643 155.406 65.374C155.429 70.735 155.285 76.08 154.351 81.364C153.31 87.28 151.794 93.055 149.023 98.452C146.984 102.422 145.482 106.722 143.068 110.435C138.359 117.674 132.839 124.309 126.634 130.373C122.089 134.814 117.114 138.648 111.763 142.037C109.872 143.234 107.956 144.412 106.181 145.769C101.13 149.627 95.3472 151.865 89.3352 153.644C87.6492 154.143 85.9952 154.743 84.3272 155.294C80.0542 156.71 75.6202 157.37 71.1822 157.952C68.4962 158.305 65.7742 158.411 63.0652 158.568C61.8111 158.64 60.5692 158.815 59.3172 158.163C58.7452 157.863 57.8312 158.217 56.9132 158.286C56.9032 157.451 57.0182 156.724 56.8522 156.069C56.6962 155.451 56.2432 154.909 55.9202 154.333C55.7602 154.427 55.6022 154.522 55.4432 154.616C55.6312 154.897 55.8812 155.155 55.9942 155.462C56.6292 157.196 55.7071 158.462 53.9202 158.04C51.1252 157.38 48.3652 156.538 45.6392 155.631C38.8642 153.374 32.9892 149.534 27.6022 144.944C23.8932 141.784 20.2392 138.51 17.3172 134.581C14.7902 131.185 12.4282 127.657 10.1592 124.083C8.62215 121.661 7.25515 119.114 6.01315 116.527C3.74915 111.816 2.49715 106.763 1.40215 101.681C0.914148 99.42 0.586152 97.123 0.256152 94.834C-0.335848 90.729 0.344149 86.719 1.26215 82.731C1.88915 80.005 2.38715 77.247 3.09015 74.541C4.13315 70.524 5.31716 66.544 6.41216 62.54C6.61716 61.786 6.68315 60.994 6.81415 60.22L6.81116 60.239ZM53.1272 59.27C53.1371 58.631 53.2292 58.354 53.1292 58.196C52.0742 56.511 51.9511 54.618 51.6702 52.711C51.1842 49.408 50.4282 46.143 49.7562 42.868C49.4862 41.549 49.1742 40.236 48.8182 38.938C48.6722 38.405 48.3592 37.917 47.9432 37.025C47.5972 37.766 47.4332 38.024 47.3512 38.305C46.8142 40.154 47.5872 41.872 47.9332 43.626C48.2551 45.262 48.8242 46.862 49.0172 48.51C49.4612 52.297 50.6861 55.757 53.1272 59.27ZM41.8812 37.513C41.3242 35.125 40.8502 32.713 40.1802 30.357C39.7642 28.899 38.9572 27.64 36.9772 27.715C37.3772 28.959 37.8622 30.049 38.0592 31.188C38.2742 32.433 38.1762 33.729 38.2992 34.995C38.4062 36.082 38.6332 37.157 38.9042 38.838C39.7622 37.256 40.5662 36.609 41.8812 37.513ZM94.7522 133.431C93.7322 132.085 92.1702 131.551 90.7132 131.127C89.8932 130.889 88.9552 131.037 88.0222 130.67C86.6612 130.133 85.1432 129.984 83.6861 129.713C83.5241 129.682 83.3111 129.92 83.1242 130.033C83.9382 131.465 83.9442 131.449 85.3372 131.939C87.7841 132.8 90.2222 133.685 92.7652 134.596C93.3121 134.274 94.0042 133.869 94.7522 133.431ZM32.2682 58.123C32.1082 58.191 31.9462 58.26 31.7862 58.328C32.1002 59.032 32.4152 59.737 32.7272 60.441C32.8702 60.381 33.0122 60.321 33.1532 60.26C32.8572 59.548 32.5622 58.835 32.2682 58.123Z" fill="black" />
							</svg>
						</div>
						
					</div>
			';
					
			if(!( 'none' == $settings['divider'] )){	
				echo tommusrhodus_svg_dividers_pluck( $settings['divider'], '' );		
			}
				
			echo '</section>';
		
		} elseif( 'standard-alt' == $settings['layout'] ){
		
			echo '
				<section class="bg-primary-alt header-inner o-hidden">
				
					<div class="container">';

						if( 'yes' == $settings['show_breadcrumbs'] || !empty( $settings['show_breadcrumbs'] ) ) {

							echo '
								<div class="row my-3">
									<div class="col">'. get_tommusrhodus_breadcrumbs() .'</div>
								</div>
							';

						}
						
						echo '
						<div class="row py-6 text-center justify-content-center align-items-center">
							<div class="col-xl-8 col-lg-10">'. $settings['content'] .'</div>
						</div>
						
					</div>
				
					<div class="decoration-wrapper d-none d-md-block">
					
				      	<div class="decoration middle-y right scale-2 d-none d-lg-block" data-jarallax-element="100">
					        <svg class="bg-primary-2" width="150" height="111" viewBox="0 0 150 111" fill="none" xmlns="http://www.w3.org/2000/svg">
					          <path d="M61.1773 84.002C60.8203 86.363 60.4453 88.721 60.1093 91.085C59.4803 95.529 58.1233 99.779 56.6563 103.996C55.9393 106.057 54.6813 106.772 52.6743 106.402C50.5633 106.011 48.3833 103.041 48.7093 100.904C48.8423 100.045 49.1252 99.203 49.3912 98.369C50.2372 95.713 51.2503 93.098 51.9363 90.4C52.7583 87.164 53.2993 83.855 53.9893 80.584C54.1353 79.889 54.2333 79.098 54.6353 78.563C55.1723 77.852 55.6393 76.871 56.8283 76.969C58.0963 77.073 59.2622 77.495 59.7622 78.703C60.4562 80.379 61.2533 82.068 61.1773 84.002Z"
					          fill="black" />
					          <path d="M141.808 65.469C141.175 64.913 140.765 64.834 140.042 65.067C138.253 65.641 136.833 65.012 135.747 63.468C135.179 62.658 134.886 61.828 135.233 60.776C136.061 58.276 136.837 55.756 137.532 53.217C138.227 50.677 138.796 48.102 139.428 45.546C140.059 42.99 140.674 40.428 141.358 37.885C141.674 36.704 142.288 36.288 143.473 36.506C144.776 36.746 146.164 36.94 146.85 38.37C146.946 38.571 147.172 38.724 147.366 38.861C148.61 39.752 148.614 40.968 148.311 42.291C147.76 44.703 147.233 47.12 146.688 49.533C145.596 54.369 144.282 59.135 142.26 63.682C142.048 64.165 141.993 64.719 141.808 65.469Z"
					          fill="black" />
					          <path d="M115.315 90.3741C115.913 92.7851 115.352 94.8371 114.647 96.9001C113.272 100.917 111.692 104.839 109.624 108.553C109.317 109.103 108.948 109.617 108.591 110.175C106.163 109.987 103.794 110.057 102.091 107.882C100.829 106.268 100.732 105.089 101.609 103.144C103.279 99.4351 104.797 95.6561 106.359 91.8981C106.634 91.2361 106.804 90.5311 107.05 89.8551C107.698 88.0641 108.087 87.8451 109.868 88.2291C113.45 89.0001 113.45 89.0001 115.315 90.3741Z"
					          fill="black" />
					          <path d="M142.858 103.827C141.477 103.947 140.307 104.139 139.139 104.133C136.707 104.119 134.553 101.671 134.787 99.167C134.875 98.227 135.178 97.291 135.488 96.39C136.715 92.842 137.98 89.322 138.584 85.58C139.07 82.564 140.246 81.965 143.105 82.99C144.138 83.359 145.13 83.839 145.779 84.881C146.361 85.82 147.152 86.628 147.746 87.363C146.821 93.161 145.323 98.598 142.858 103.827ZM141.942 102.875C141.932 102.992 141.922 103.11 141.915 103.229C141.954 103.174 142.02 103.122 142.022 103.065C142.026 103.003 141.972 102.938 141.942 102.875Z"
					          fill="black" />
					          <path d="M82.239 99.639C80.661 99.499 79.382 99.506 78.159 99.248C76.194 98.835 74.6139 95.684 75.2569 93.631C75.6379 92.416 76.132 91.235 76.478 90.01C77.603 86.026 78.6749 82.027 79.7829 78.037C80.1249 76.814 80.488 75.696 81.828 75.017C83.947 73.945 85.935 74.667 86.64 76.881C87.38 79.204 86.994 81.491 86.452 83.718C85.761 86.566 85.4149 89.523 84.0849 92.195C84.3619 94.85 81.952 96.76 82.239 99.639Z"
					          fill="black" />
					          <path d="M110.825 71.084C110.555 71.113 110.263 71.056 110.096 71.176C108.407 72.391 106.922 71.778 105.551 70.641C104.139 69.471 103.174 68.05 103.572 66.075C104.027 63.813 104.529 61.559 104.945 59.29C105.431 56.628 105.998 53.965 106.242 51.276C106.459 48.915 107.537 48.219 109.752 48.966C110.957 49.372 112.176 49.621 113.098 50.646C114.399 52.096 114.852 53.55 114.545 55.615C113.945 59.646 113.09 63.603 112.07 67.533C111.774 68.676 111.292 69.773 110.825 71.084Z"
					          fill="black" />
					          <path d="M29.6628 97.127C29.0538 97.059 28.3948 96.811 27.8388 96.957C26.3758 97.34 25.4428 96.774 24.5518 95.678C23.5418 94.433 22.9468 93.131 23.2468 91.512C23.7858 88.616 24.2608 85.705 24.8858 82.826C25.5598 79.717 26.3548 76.634 27.1298 73.547C27.6918 71.305 29.4738 70.629 31.2548 72.123C33.0708 73.646 34.2628 75.403 33.6338 78.129C32.6028 82.608 31.9328 87.172 31.0868 91.695C30.9868 92.236 30.7098 92.742 30.5728 93.279C30.2668 94.49 29.9948 95.712 29.6628 97.127Z"
					          fill="black" />
					          <path d="M149.825 8.36506C147.995 13.5131 146.27 18.5731 144.389 23.5751C143.356 26.3271 139.018 27.3001 136.752 25.4291C135.752 24.6031 135.547 23.8961 135.982 22.7171C137.916 17.4741 139.841 12.2271 141.8 6.99309C142.124 6.12709 142.21 4.92908 143.333 4.73808C144.616 4.51908 145.851 4.97209 146.882 5.78209C147.503 6.27109 148.027 6.88406 148.611 7.42106C148.903 7.68806 149.235 7.91006 149.825 8.36506Z"
					          fill="black" />
					          <path d="M8.46604 24.099C6.98004 24.167 5.87605 24.327 4.79005 24.241C3.44005 24.134 2.48503 23.2819 1.70603 22.2009C1.01303 21.2379 1.21404 20.338 1.55604 19.281C2.63204 15.954 3.67105 12.609 4.52705 9.21996C5.01105 7.30096 5.12103 5.28497 5.36903 3.30797C5.56603 1.73297 6.36505 1.01995 7.87705 1.23995C9.52205 1.47895 10.8341 2.17097 11.5451 3.83697C12.2751 5.54897 12.867 7.21295 11.926 9.02395C11.854 9.16295 11.783 9.32796 11.789 9.47796C11.92 12.868 10.435 15.883 9.59605 19.049C9.15905 20.698 8.84704 22.379 8.46604 24.099Z"
					          fill="black" />
					          <path d="M82.2708 56.719C81.2398 56.858 80.3878 56.842 79.6298 57.104C78.3778 57.535 77.4538 57.124 76.6548 56.239C75.5398 54.999 74.9708 53.719 75.5828 51.912C77.1898 47.153 78.2628 42.255 78.8328 37.254C78.9128 36.544 79.0598 35.837 79.2408 35.145C79.5548 33.938 80.3068 33.456 81.5198 33.613C84.2448 33.964 86.1588 36.695 85.6648 39.43C84.9848 43.186 84.4028 46.961 83.6938 50.711C83.3128 52.72 82.7548 54.697 82.2708 56.719Z"
					          fill="black" />
					          <path d="M7.60279 107.088C6.34679 107.162 5.5558 106.785 5.0908 105.654C4.7368 104.791 4.07078 104.043 3.48578 103.289C2.33178 101.804 2.2648 100.262 3.0308 98.581C3.9218 96.625 4.7768 94.65 5.5698 92.652C6.1228 91.256 6.55579 89.811 7.01679 88.381C7.16279 87.928 7.21379 87.445 7.33079 86.983C7.77379 85.249 8.5338 84.391 9.6648 84.329C11.0538 84.252 12.7608 85.518 13.1258 87.064C14.2548 91.847 13.6008 96.364 10.9808 100.569C10.7078 101.01 10.6388 101.577 10.4648 102.083C10.3668 102.374 10.2908 102.685 10.1348 102.944C9.2958 104.346 8.43279 105.734 7.60279 107.088Z"
					          fill="black" />
					          <path d="M28.5579 35.457C30.9369 34.317 32.9469 34.942 34.0289 37.084C35.3839 39.766 35.5128 39.787 34.9938 42.934C34.3318 46.944 33.4429 50.917 32.6289 54.901C32.5509 55.274 32.3929 55.758 32.1109 55.938C31.2519 56.488 31.2479 57.364 31.1849 58.204C30.3689 58.147 29.6769 58.067 28.9859 58.058C27.4879 58.035 26.9178 57.601 26.3258 56.222C25.7848 54.956 26.0559 53.706 26.2299 52.451C26.8589 47.95 27.5149 43.453 28.1459 38.952C28.3119 37.771 28.4269 36.584 28.5579 35.457Z"
					          fill="black" />
					          <path d="M114.899 7.67801C113.784 12.809 112.727 17.677 111.684 22.47C110.725 22.741 109.776 22.97 108.852 23.278C107.248 23.811 106.096 22.954 104.993 22.04C104.011 21.226 103.429 20.16 103.905 18.864C105.426 14.721 105.913 10.354 106.768 6.06601C107.282 3.49601 107.768 3.198 110.321 3.698C112.063 4.039 113.311 5.07299 114.233 6.54599C114.522 7.00999 114.79 7.49101 114.899 7.67801Z"
					          fill="black" />
					          <path d="M54.4378 63.111C53.8558 63.111 53.3908 63.082 52.9298 63.117C49.9998 63.332 47.9808 61.1211 48.4528 58.1871C48.5148 57.7961 48.6698 57.419 48.7868 57.039C50.0528 52.908 51.0058 48.708 51.5898 44.426C51.6538 43.951 51.7228 43.476 51.8188 43.007C52.2078 41.115 52.7228 40.877 54.6258 41.046C56.7628 41.235 58.8118 44.015 58.4168 46.016C57.8778 48.756 57.5338 51.539 56.8988 54.255C56.2818 56.886 55.3968 59.454 54.6388 62.052C54.5538 62.35 54.5218 62.664 54.4378 63.111Z"
					          fill="black" />
					          <path d="M10.5169 53.2879C9.29995 54.7029 8.38996 56.2669 8.22996 58.1709C8.15596 59.0409 7.72395 59.455 6.85295 59.479C6.05795 59.499 5.26693 59.6 4.47193 59.613C2.04793 59.651 0.125963 57.2889 0.725963 54.9639C0.944963 54.1209 1.32995 53.318 1.65995 52.506C2.76695 49.77 3.86696 47.03 5.00996 44.308C5.41996 43.331 5.40295 41.9199 6.85395 41.7969C8.41595 41.6639 9.98294 41.9969 10.9729 43.2909C11.7849 44.3529 12.338 45.6309 12.889 46.8649C13.057 47.2389 12.953 47.838 12.772 48.243C12.129 49.681 11.3949 51.0779 10.7099 52.4979C10.5909 52.7479 10.552 53.035 10.476 53.306L10.5169 53.2879Z"
					          fill="black" />
					          <path d="M60.8919 3.11896C60.4209 5.22596 59.9579 7.35097 59.4699 9.46997C58.7239 12.711 57.9759 15.951 57.1989 19.185C56.9939 20.04 56.7789 20.898 55.4759 20.508C55.0439 20.379 54.4719 20.83 53.9509 20.873C51.8319 21.05 50.2629 19.777 50.0659 17.679C49.9999 16.985 50.0699 16.242 50.2479 15.566C51.1369 12.181 52.1169 8.81997 52.9899 5.43197C53.3609 3.99097 54.2009 2.86196 55.0969 1.75696C55.9509 0.704955 57.1149 0.492976 58.2729 1.20398C59.1419 1.73598 59.9289 2.40596 60.8919 3.11896Z"
					          fill="black" />
					          <path d="M32.0622 18.568C30.9982 18.568 30.2012 18.489 29.4252 18.588C28.4212 18.716 27.7572 18.316 27.1692 17.562C25.9522 16 25.5932 14.433 26.3802 12.469C27.7162 9.13599 28.8492 5.719 29.9722 2.307C30.3532 1.153 30.8742 0.0879904 32.0952 0.0119904C33.3492 -0.0660096 34.5422 0.375987 35.4132 1.53299C36.7902 3.36299 37.4272 5.18201 36.5052 7.47101C35.2842 10.5 34.2822 13.616 33.1362 16.677C32.8902 17.338 32.4372 17.921 32.0622 18.568Z"
					          fill="black" />
					          <path d="M82.4332 24.1381C81.4762 24.2461 80.5192 24.2991 79.5852 24.4711C77.0972 24.9311 75.0562 22.8111 75.6652 20.3251C75.7962 19.7881 76.0792 19.2901 76.2762 18.7681C77.4302 15.7121 78.5902 12.6581 79.7292 9.59706C80.5002 7.52106 80.9012 7.28206 83.1162 7.73506C83.8932 7.89406 84.6692 8.07806 85.4362 8.27906C85.7392 8.35806 86.0221 8.51406 86.2741 8.61906C87.0711 10.1071 86.9302 11.5381 86.4112 13.0101C85.6412 15.1821 84.8782 17.3591 84.2122 19.5631C83.7592 21.0621 83.7022 22.6941 82.4332 24.1381Z"
					          fill="black" />
					        </svg>

					      </div>
					      <div class="decoration middle-y right blend-mode-multiply d-none d-xl-block" data-jarallax-element="50 50">
					        <svg class="bg-primary" width="338" height="277" viewBox="0 0 338 277" fill="none" xmlns="http://www.w3.org/2000/svg">
					          <path d="M136.018 0.775024C143.338 0.998024 150.311 2.86002 157.217 4.90402C161.951 6.30502 166.533 8.21602 171.238 9.72702C177.683 11.799 184.205 13.642 190.654 15.704C198.047 18.067 205.496 20.302 212.734 23.077C219.181 25.549 225.818 26.16 232.576 26.624C242.613 27.313 252.408 29.541 262.14 31.958C267.613 33.318 273.015 35.013 278.376 36.777C286.159 39.338 292.769 43.771 298.435 49.705C300.869 52.253 303.482 54.662 306.224 56.875C310.91 60.658 314.185 65.568 317.597 70.391C317.999 70.957 318.31 71.699 318.861 72.031C323.925 75.085 326.72 80.024 329.47 84.928C331.605 88.738 333.45 92.72 335.236 96.711C335.974 98.361 336.533 100.215 336.629 102.006C336.979 108.465 337.936 114.881 337.352 121.411C336.889 126.604 336.916 131.868 337.11 137.086C337.676 152.284 335.641 167.235 333.401 182.2C331.815 192.802 330.878 203.502 329.278 214.101C328.417 219.807 327.28 225.578 325.321 230.976C323.759 235.279 321.196 239.409 318.317 243.006C311.585 251.42 303.104 257.68 292.893 261.414C288.381 263.064 283.952 265.016 279.332 266.275C273.076 267.98 266.711 269.338 260.33 270.509C250.605 272.292 240.844 273.878 231.07 275.381C220.672 276.98 210.306 277.306 199.939 274.719C194.33 273.32 188.527 272.723 182.869 271.504C166.828 268.049 151.043 263.651 135.754 257.669C130.918 255.776 126.25 253.478 122.199 249.956C118.49 246.731 113.928 244.469 110.316 241.155C103.357 234.766 96.6579 228.074 90.1249 221.245C84.3729 215.231 79.0449 208.814 73.4259 202.671C71.6229 200.7 69.3989 199.121 67.5219 197.212C61.8789 191.478 56.3579 185.624 50.6959 179.909C48.0139 177.202 45.0629 174.763 42.3439 172.091C39.7309 169.523 37.2799 166.791 34.7229 164.164C30.1899 159.507 25.8419 154.642 21.0319 150.288C14.4459 144.325 9.29194 137.288 4.85794 129.733C1.90494 124.702 0.404932 119.126 0.994932 113.109C1.35393 109.453 1.56894 105.873 3.02594 102.364C4.82294 98.043 7.59594 94.544 11.0199 91.581C16.4609 86.871 22.0179 82.28 27.7129 77.881C34.4159 72.703 41.2719 67.718 48.1519 62.774C53.0819 59.232 58.3649 56.157 63.1269 52.411C72.1059 45.348 81.2339 38.467 89.4079 30.405C96.0349 23.868 102.898 17.54 110.002 11.527C115.279 7.06004 121.135 3.23104 128.049 1.65704C130.639 1.07104 133.357 1.05302 136.018 0.775024ZM19.8459 102.8C15.5139 101.001 13.7579 101.522 12.1429 105.364C13.5239 105.867 14.8829 106.363 16.5709 106.978C16.7739 105.683 16.8949 104.912 16.9929 104.287C17.9989 103.763 18.9229 103.281 19.8479 102.799C21.2859 101.622 23.0749 100.717 23.4099 98.469C20.4119 98.883 20.4119 98.883 19.8459 102.8ZM118.352 15.815C117.153 17.925 116.342 19.402 117.231 21.328C119.746 19.487 119.773 19.382 118.352 15.815ZM36.2909 86.69C35.4119 88.799 34.8089 90.248 34.0939 91.961C37.8889 90.785 37.8889 90.785 36.2909 86.69ZM129.395 162.873C128.641 162.383 128.006 161.799 127.858 161.903C127.292 162.306 126.881 162.927 126.413 163.468C126.843 163.712 127.337 164.224 127.684 164.138C128.211 164.009 128.639 163.465 129.395 162.873ZM137.797 163.645C137.248 164.305 136.658 164.709 136.697 165.036C136.763 165.591 137.228 166.097 137.525 166.623C137.986 166.255 138.761 165.928 138.818 165.505C138.881 165.033 138.287 164.477 137.797 163.645ZM137.221 207.492C137.242 207.855 137.264 208.219 137.285 208.582C138.129 208.456 138.973 208.33 139.816 208.205C139.787 207.967 139.757 207.73 139.73 207.492C138.895 207.492 138.057 207.492 137.221 207.492ZM110.674 30.56C110.768 30.297 110.862 30.035 110.957 29.772C110.123 29.451 109.291 29.13 108.457 28.809C108.357 29.097 108.256 29.386 108.154 29.674C108.994 29.969 109.834 30.265 110.674 30.56ZM116.773 160.416C116.58 160.891 116.285 161.258 116.357 161.528C116.435 161.827 116.851 162.037 117.121 162.285C117.336 161.902 117.652 161.535 117.713 161.129C117.736 160.968 117.193 160.722 116.773 160.416ZM124.658 162.574C123.793 162.347 123.324 162.142 122.863 162.152C122.707 162.156 122.562 162.708 122.414 163.009C122.768 163.15 123.127 163.408 123.473 163.392C123.754 163.381 124.02 163.036 124.658 162.574ZM133.973 165.672C133.819 165.484 133.664 165.297 133.51 165.11C133.348 165.387 133.151 165.654 133.059 165.954C133.039 166.011 133.434 166.196 133.637 166.322C133.748 166.105 133.861 165.89 133.973 165.672ZM115.15 24.039C114.955 23.876 114.759 23.714 114.566 23.552C114.468 23.778 114.254 24.034 114.302 24.223C114.353 24.418 114.656 24.549 114.849 24.708C114.949 24.486 115.051 24.263 115.15 24.039Z"
					          fill="black" />
					        </svg>

				     	</div>
						
					</div>
			';
					
			if(!( 'none' == $settings['divider'] )){	
				echo tommusrhodus_svg_dividers_pluck( $settings['divider'], '' );		
			}
				
			echo '</section>';
		
		} elseif( 'dark-background' == $settings['layout'] ) {
			
			echo '
				<section class="bg-primary-3 header-inner p-0 jarallax position-relative o-hidden" data-overlay>
				
					<div class="container py-0 layer-2">';

						if( 'yes' == $settings['show_breadcrumbs'] || !empty( $settings['show_breadcrumbs'] ) ) {

							echo '
								<div class="row my-3 text-light">
									<div class="col">'. get_tommusrhodus_breadcrumbs() .'</div>
								</div>
							';

						}
						
						echo '						
						<div class="row my-4 my-md-6 text-light" data-aos="fade-up">
							<div class="col-lg-9 col-xl-6">'. $settings['content'] .'</div>
						</div>

					</div>
					
					<div class="decoration right middle-y scale-4">
						<svg class="bg-primary" width="181" height="248" viewBox="0 0 181 248" fill="none" xmlns="http://www.w3.org/2000/svg">
							<path d="M173.496 137.406C174.74 140.125 174.201 142.961 172.992 145.955C171.785 148.945 171.244 152.203 170.371 155.332C169.545 158.285 169.023 161.391 167.74 164.135C164.455 171.167 160.851 178.053 157.302 184.959C156.15 187.203 154.769 189.328 153.507 191.517C150.778 196.252 146.98 200.155 143.411 204.207C139.247 208.934 134.907 213.528 130.36 217.884C126.981 221.119 123.356 224.19 119.477 226.79C110.78 232.623 101.875 238.142 91.7499 241.311C88.3119 242.386 84.9839 243.811 81.5509 244.904C73.1449 247.579 64.4319 248.129 55.8069 247.077C43.0409 245.519 34.2129 238.431 30.1099 226.075C28.8639 222.324 28.1059 218.372 27.5139 214.454C26.8169 209.821 26.5859 205.12 26.0709 200.456C25.3349 193.802 24.3459 187.22 21.4659 181.048C18.4739 174.631 16.5169 167.857 15.4639 160.864C14.0029 151.18 11.2059 141.854 8.4269 132.5C6.2689 125.236 4.1689 117.936 3.2239 110.374C3.0229 108.764 2.7159 107.161 2.3649 105.576C1.5039 101.679 1.03089 97.749 1.08989 93.75C1.11089 92.31 1.1309 90.823 0.8169 89.433C-0.1261 85.264 0.427893 81.211 1.17589 77.083C2.60189 69.223 4.9509 61.659 7.8319 54.234C11.6759 44.329 17.4339 35.662 24.7729 28.024C27.4019 25.289 30.1459 22.662 32.7379 19.892C41.5149 10.511 52.7829 5.93202 64.9919 3.34302C76.9139 0.815015 89.0599 0.543025 101.144 1.31003C118.3 2.39903 134.746 6.51404 149.367 15.95C158.662 21.948 165.805 29.942 169.615 40.639C172.851 49.726 174.459 59.13 175.758 68.625C176.248 72.201 176.719 75.792 177.459 79.32C179.471 88.912 180.406 98.573 180.119 108.383C179.885 116.393 178.273 124.157 176.322 131.861C175.854 133.709 174.568 135.353 173.496 137.406ZM114.193 74.056C116.129 74.66 116.129 74.66 116.709 73.132C115.756 73.482 114.975 73.769 114.193 74.056Z" fill="black" />
						</svg>
					</div>
					
				</section>
			';
			
		} elseif( 'dark-background-alert' == $settings['layout'] ) {
			
			echo '
				<section class="bg-primary-3 header-inner p-0 jarallax position-relative o-hidden" data-overlay>
				
					<div class="container py-0 layer-2">';

						if( 'yes' == $settings['show_breadcrumbs'] || !empty( $settings['show_breadcrumbs'] ) ) {

							echo '
								<div class="row my-3 text-light">
									<div class="col">'. get_tommusrhodus_breadcrumbs() .'</div>
								</div>
							';

						}
						
						echo '
						<div class="row my-4 my-md-6 text-light" data-aos="fade-up">
							<div class="col-lg-9 col-xl-6">'. $settings['content'] .'</div>
						</div>

						<div class="alert alert-primary mb-5">
				          <div class="d-flex">
				            <svg class="icon bg-primary" width="24px" height="24px" viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
				              <title>Icon For Selected-file</title>
				              <g stroke="none" stroke-width="1" fill="none" fill-rule="evenodd">
				                <polygon points="0 0 24 0 24 24 0 24" opacity="0"></polygon>
				                <path d="M4.85714286,1 L11.7364114,1 C12.0910962,1 12.4343066,1.12568431 12.7051108,1.35473959 L17.4686994,5.3839416 C17.8056532,5.66894833 18,6.08787823 18,6.52920201 L18,19.0833333 C18,20.8738751 17.9795521,21 16.1428571,21 L4.85714286,21 C3.02044787,21 3,20.8738751 3,19.0833333 L3,2.91666667 C3,1.12612489 3.02044787,1 4.85714286,1 Z M8,12 C7.44771525,12 7,12.4477153 7,13 C7,13.5522847 7.44771525,14 8,14 L15,14 C15.5522847,14 16,13.5522847 16,13 C16,12.4477153 15.5522847,12 15,12 L8,12 Z M8,16 C7.44771525,16 7,16.4477153 7,17 C7,17.5522847 7.44771525,18 8,18 L11,18 C11.5522847,18 12,17.5522847 12,17 C12,16.4477153 11.5522847,16 11,16 L8,16 Z" fill="#000000" fill-rule="nonzero" opacity="0.3"></path>
				                <path d="M6.85714286,3 L14.7364114,3 C15.0910962,3 15.4343066,3.12568431 15.7051108,3.35473959 L20.4686994,7.3839416 C20.8056532,7.66894833 21,8.08787823 21,8.52920201 L21,21.0833333 C21,22.8738751 20.9795521,23 19.1428571,23 L6.85714286,23 C5.02044787,23 5,22.8738751 5,21.0833333 L5,4.91666667 C5,3.12612489 5.02044787,3 6.85714286,3 Z M8,12 C7.44771525,12 7,12.4477153 7,13 C7,13.5522847 7.44771525,14 8,14 L15,14 C15.5522847,14 16,13.5522847 16,13 C16,12.4477153 15.5522847,12 15,12 L8,12 Z M8,16 C7.44771525,16 7,16.4477153 7,17 C7,17.5522847 7.44771525,18 8,18 L11,18 C11.5522847,18 12,17.5522847 12,17 C12,16.4477153 11.5522847,16 11,16 L8,16 Z" fill="#000000" fill-rule="nonzero"></path>
				              </g>
				            </svg>
				            <div class="ml-2 flex-grow-1">
				              '. $settings['alert_content'] .'
				            </div>
				          </div>
				        </div>

					</div>
					
					<div class="decoration right middle-y scale-4">
						<svg class="bg-primary" width="181" height="248" viewBox="0 0 181 248" fill="none" xmlns="http://www.w3.org/2000/svg">
							<path d="M173.496 137.406C174.74 140.125 174.201 142.961 172.992 145.955C171.785 148.945 171.244 152.203 170.371 155.332C169.545 158.285 169.023 161.391 167.74 164.135C164.455 171.167 160.851 178.053 157.302 184.959C156.15 187.203 154.769 189.328 153.507 191.517C150.778 196.252 146.98 200.155 143.411 204.207C139.247 208.934 134.907 213.528 130.36 217.884C126.981 221.119 123.356 224.19 119.477 226.79C110.78 232.623 101.875 238.142 91.7499 241.311C88.3119 242.386 84.9839 243.811 81.5509 244.904C73.1449 247.579 64.4319 248.129 55.8069 247.077C43.0409 245.519 34.2129 238.431 30.1099 226.075C28.8639 222.324 28.1059 218.372 27.5139 214.454C26.8169 209.821 26.5859 205.12 26.0709 200.456C25.3349 193.802 24.3459 187.22 21.4659 181.048C18.4739 174.631 16.5169 167.857 15.4639 160.864C14.0029 151.18 11.2059 141.854 8.4269 132.5C6.2689 125.236 4.1689 117.936 3.2239 110.374C3.0229 108.764 2.7159 107.161 2.3649 105.576C1.5039 101.679 1.03089 97.749 1.08989 93.75C1.11089 92.31 1.1309 90.823 0.8169 89.433C-0.1261 85.264 0.427893 81.211 1.17589 77.083C2.60189 69.223 4.9509 61.659 7.8319 54.234C11.6759 44.329 17.4339 35.662 24.7729 28.024C27.4019 25.289 30.1459 22.662 32.7379 19.892C41.5149 10.511 52.7829 5.93202 64.9919 3.34302C76.9139 0.815015 89.0599 0.543025 101.144 1.31003C118.3 2.39903 134.746 6.51404 149.367 15.95C158.662 21.948 165.805 29.942 169.615 40.639C172.851 49.726 174.459 59.13 175.758 68.625C176.248 72.201 176.719 75.792 177.459 79.32C179.471 88.912 180.406 98.573 180.119 108.383C179.885 116.393 178.273 124.157 176.322 131.861C175.854 133.709 174.568 135.353 173.496 137.406ZM114.193 74.056C116.129 74.66 116.129 74.66 116.709 73.132C115.756 73.482 114.975 73.769 114.193 74.056Z" fill="black" />
						</svg>
					</div>
					
				</section>
			';
			
		} elseif( 'image-background' == $settings['layout'] ) {
			
			echo '
				<section data-jarallax data-speed="0.2" class="bg-dark text-light header-inner p-0 jarallax o-hidden" data-overlay>
					
					'. wp_get_attachment_image( $settings['image']['id'], 'full', 0, array( 'class' => 'jarallax-img opacity-30' ) ) .'
			      
					<div class="container py-0 layer-2">';

						if( 'yes' == $settings['show_breadcrumbs'] || !empty( $settings['show_breadcrumbs'] ) ) {

							echo '
								<div class="row my-3">
									<div class="col">'. get_tommusrhodus_breadcrumbs() .'</div>
								</div>
							';

						}
						
						echo '
						<div class="row my-4 my-md-6" data-aos="fade-up">
							<div class="col-lg-9 col-xl-8">'. $settings['content'] .'</div>
						</div>
					</div>
			';
			
			if(!( 'none' == $settings['divider'] )){	
				echo tommusrhodus_svg_dividers_pluck( $settings['divider'], '' );		
			}
				
			echo '</section>';
			
		} elseif( 'fullscreen' == $settings['layout'] ) {
			
			echo '
				<section class="bg-primary text-light min-vh-100 o-hidden d-flex flex-column justify-content-center" data-overlay>
			      	<div class="container">
			        	<div class="row justify-content-center text-center align-items-center">
			          		<div class="col-xl-8 col-lg-9 col-md-10 layer-3" data-aos="fade-up" data-aos-delay="500">
			            		'. $settings['content'] .'
				          	</div>
				        </div>
			      	</div>
			      	<div>
				        <div class="decoration right middle-y scale-3">
				          <svg class="bg-primary-2" width="298" height="197" viewBox="0 0 298 197" fill="none" xmlns="http://www.w3.org/2000/svg">
				            <path d="M271.518 116.857C266.116 125.511 259.584 133.287 253.194 141.164C248.36 147.125 243.548 153.103 238.583 158.953C236.134 161.84 233.362 164.453 230.733 167.185C229.881 168.072 228.921 168.871 228.172 169.833C225.727 172.979 222.572 175.452 220.145 178.651C217.581 182.032 213.84 184.145 210.204 186.288C201.958 191.145 193.024 193.809 183.61 195.366C174.13 196.932 164.633 196.987 155.102 196.749C148.211 196.575 141.723 194.466 135.547 191.72C126.522 187.704 117.201 184.554 107.795 181.695C102.133 179.974 96.211 179.015 90.348 178.072C82.455 176.801 74.483 176.021 66.59 174.748C61.49 173.924 56.395 173.656 51.27 173.844C41.399 174.205 31.62 172.856 21.799 172.319C17.233 172.069 12.623 171.329 8.32199 169.428C3.47399 167.289 0.64998 163.86 0.50198 158.356C0.33398 152.055 1.14497 145.872 2.36097 139.725C4.05597 131.147 6.94698 122.92 9.63998 114.625C11.435 109.1 14.044 104.068 17.138 99.191C20.263 94.261 22.833 88.964 26.156 84.181C28.943 80.169 32.218 76.415 35.73 73.013C39.232 69.62 43.835 67.853 48.488 66.47C49.535 66.159 50.73 66.247 51.851 66.298C56.263 66.493 60.572 67.214 64.898 68.22C71.472 69.749 77.906 72.04 84.709 72.466C87.564 72.646 90.438 72.616 93.301 72.558C98.117 72.46 102.93 72.236 107.746 72.117C113 71.986 117.902 70.75 122.18 67.582C129.756 61.973 137.328 56.355 144.856 50.678C150.336 46.543 155.829 42.416 161.157 38.088C167.837 32.662 174.261 26.918 180.968 21.529C186.708 16.916 192.419 12.265 198.823 8.50796C202.481 6.35996 206.471 5.08997 210.255 3.29897C212.142 2.40397 214.415 2.16795 216.544 1.97295C222.355 1.43795 228.177 0.931955 234.007 0.701955C239.169 0.498955 244.191 1.50895 249.097 3.18795C256.88 5.85395 264.337 9.12595 271.404 13.417C275.808 16.089 280.133 18.696 283.58 22.601C285.338 24.591 287.455 26.2709 289.187 28.2829C296.781 37.0899 298.767 47.203 296.302 58.465C295.044 64.211 293.189 69.723 290.986 75.139C287.625 83.399 284.443 91.746 280.726 99.846C278.041 105.691 274.613 111.197 271.518 116.857ZM95.393 132.113C95.569 132.066 95.745 132.021 95.92 131.974C95.768 131.382 95.617 130.788 95.465 130.197C95.283 130.244 95.102 130.291 94.922 130.338C95.078 130.93 95.236 131.521 95.393 132.113ZM225.68 158.404C225.83 158.343 225.985 158.293 226.121 158.211C226.142 158.199 226.103 157.986 226.062 157.977C225.916 157.94 225.755 157.948 225.601 157.938C225.627 158.092 225.652 158.248 225.68 158.404Z"
				            fill="black" />
				          </svg>

				        </div>
				        <div class="decoration right bottom scale-3">
				          <svg class="bg-white" width="114" height="64" viewBox="0 0 114 64" fill="none" xmlns="http://www.w3.org/2000/svg">
				            <path d="M73.32 8.86801C73.459 7.62901 72.664 7.25601 71.906 6.76901C70.867 6.10101 70.59 5.277 71.031 4.345C71.222 3.945 71.58 3.52399 71.969 3.33499C73.746 2.47299 76.209 3.20301 77.276 4.85901C77.698 5.51701 77.764 6.25802 77.266 6.87702C76.352 8.01402 75.258 8.90001 73.32 8.86801Z"
				            fill="black" />
				            <path d="M33.5251 51.6231C32.8901 51.3521 32.2261 51.1311 31.6251 50.7981C30.5961 50.2291 29.8591 49.4331 29.9791 48.1321C30.0831 46.9981 31.0281 46.0351 32.3131 45.7621C32.5471 45.7121 32.7801 45.6531 33.0181 45.6251C34.1571 45.4901 34.8851 45.9071 35.2481 46.9691C35.5371 47.8111 35.6291 48.6821 35.3671 49.5591C35.1191 50.3861 34.8261 50.7141 33.5251 51.6231Z"
				            fill="black" />
				            <path d="M3.47102 38.293C3.28302 38.205 2.79302 38.133 2.59002 37.854C1.75002 36.698 0.916028 35.502 0.709028 34.047C0.582028 33.168 1.12103 32.389 1.85003 32.159C2.72303 31.885 3.61003 31.74 4.24803 32.605C4.86303 33.436 5.44502 34.295 5.98002 35.179C6.42302 35.909 6.49802 36.75 5.98002 37.446C5.43202 38.186 4.63102 38.556 3.47102 38.293Z"
				            fill="black" />
				            <path d="M67.549 22.483C67.445 22.721 67.2249 23.225 67.0039 23.7311C66.5509 24.7751 64.135 25.134 63.51 24.034C63.164 23.424 62.795 22.809 62.571 22.15C62.307 21.368 62.7659 20.481 63.5589 20.271C64.3169 20.071 65.121 19.966 65.907 19.967C67.271 19.969 67.865 20.8 67.549 22.483Z"
				            fill="black" />
				            <path d="M113.014 61.366C112.778 62.277 112.307 62.954 111.313 63.154C110.467 63.323 109.659 63.007 109.295 62.258C108.984 61.621 108.719 60.96 108.49 60.29C108.24 59.564 108.351 58.879 108.888 58.29C109.443 57.681 110.148 57.632 110.87 57.811C112.268 58.155 113.34 59.966 113.014 61.366Z"
				            fill="black" />
				            <path d="M77.145 37.633C77.024 37.16 77.006 36.559 76.731 36.115C76.016 34.961 76.557 34.14 77.311 33.351C77.944 32.688 78.721 32.651 79.52 32.996C80.547 33.439 81.223 34.154 81.262 35.35C81.305 36.698 80.594 37.668 79.283 37.959C78.568 38.118 77.859 38.126 77.145 37.633ZM79.16 34.699C79.01 34.366 78.969 34.112 78.818 33.978C78.531 33.719 78.056 33.958 77.957 34.434C77.918 34.622 78.053 34.848 78.107 35.057C78.438 34.945 78.768 34.833 79.16 34.699Z"
				            fill="black" />
				            <path d="M20.1408 2.24493C19.9418 3.29493 19.8318 4.24094 19.5608 5.13894C19.4088 5.63894 19.0378 6.10193 18.6738 6.49993C18.2758 6.93693 17.7188 7.04993 17.1448 6.84093C16.5428 6.62293 16.3168 6.10692 16.2898 5.53292C16.2688 5.06392 16.3778 4.58994 16.4228 4.11794C16.4678 3.64594 16.5828 3.16294 16.5228 2.70394C16.4528 2.16194 16.1598 1.58792 16.8118 1.25292C17.4618 0.918921 18.1768 0.801922 18.8448 1.22292C19.2418 1.47292 19.5878 1.80293 20.1408 2.24493Z"
				            fill="black" />
				            <path d="M23.2341 25.618C24.1191 26.151 24.435 26.944 24.58 27.828C24.73 28.752 24.307 29.665 23.58 30.052C22.793 30.472 22.0041 30.371 21.3551 29.843C20.8821 29.458 20.466 28.929 20.201 28.379C19.627 27.184 20.1251 26.191 21.4371 25.85C22.0351 25.693 22.6721 25.686 23.2341 25.618Z"
				            fill="black" />
				            <path d="M53.8851 37.898C53.8231 38.16 53.7891 38.316 53.7461 38.469C53.3441 39.94 52.426 40.185 51.365 39.093C51.033 38.751 50.7011 38.396 50.4451 37.997C49.6231 36.716 50.181 35.248 51.636 34.79C52.845 34.41 53.9041 35.085 53.9821 36.337C54.0181 36.89 53.9161 37.449 53.8851 37.898Z"
				            fill="black" />
				            <path d="M96.3362 23.504C96.6952 24.849 96.9922 25.958 97.3112 27.15C96.9012 27.557 96.5022 28.055 96.0062 28.419C95.2012 29.007 94.0222 28.583 93.6902 27.628C93.3722 26.714 93.1762 25.818 93.8382 24.92C94.4672 24.064 95.0642 23.255 96.3362 23.504Z" fill="black"
				            />
				            <path d="M107.16 42.976C106.738 42.19 106.299 41.369 105.873 40.572C106.926 38.66 107.978 38.166 109.469 38.85C110.057 39.12 110.299 39.577 110.153 40.205C109.954 41.068 109.678 41.916 108.983 42.514C108.492 42.935 107.914 43.205 107.16 42.976Z" fill="black"
				            />
				            <path d="M45.7482 13.598C45.4982 14.83 43.6893 15.898 42.5393 15.523C42.3933 15.475 42.2343 15.396 42.1313 15.286C41.5783 14.695 41.5333 12.985 42.0393 12.364C42.7253 11.522 44.2422 11.273 45.0762 11.93C45.6042 12.346 45.8182 12.954 45.7482 13.598Z" fill="black"
				            />
				            <path d="M62.6021 60.476C60.2951 60.503 60.295 60.503 60.123 59.02C60.377 58.64 60.6561 58.923 60.9351 58.978C61.4471 59.076 61.8301 58.914 62.1091 58.435C62.6521 57.508 62.67 57.518 63.929 57.089C64.077 57.409 64.3251 57.734 64.3701 58.084C64.4811 58.947 64.5461 59.822 64.5461 60.692C64.5461 61.367 64.2671 61.915 63.3231 62.044C63.0881 61.533 62.8401 60.995 62.6021 60.476Z"
				            fill="black" />
				            <path d="M84.02 54.013C84.163 54.756 83.719 55.267 83.211 55.602C82.498 56.072 81.695 56.422 80.899 56.745C80.362 56.963 79.821 56.411 79.895 55.8C80.032 54.663 80.752 53.941 81.704 53.484C82.557 53.073 83.383 53.232 84.02 54.013Z" fill="black" />
				          </svg>

				        </div>
				        <div class="decoration top left scale-2  d-none d-md-block">
				          <svg class="bg-primary-3" width="338" height="277" viewBox="0 0 338 277" fill="none" xmlns="http://www.w3.org/2000/svg">
				            <path d="M136.018 0.775024C143.338 0.998024 150.311 2.86002 157.217 4.90402C161.951 6.30502 166.533 8.21602 171.238 9.72702C177.683 11.799 184.205 13.642 190.654 15.704C198.047 18.067 205.496 20.302 212.734 23.077C219.181 25.549 225.818 26.16 232.576 26.624C242.613 27.313 252.408 29.541 262.14 31.958C267.613 33.318 273.015 35.013 278.376 36.777C286.159 39.338 292.769 43.771 298.435 49.705C300.869 52.253 303.482 54.662 306.224 56.875C310.91 60.658 314.185 65.568 317.597 70.391C317.999 70.957 318.31 71.699 318.861 72.031C323.925 75.085 326.72 80.024 329.47 84.928C331.605 88.738 333.45 92.72 335.236 96.711C335.974 98.361 336.533 100.215 336.629 102.006C336.979 108.465 337.936 114.881 337.352 121.411C336.889 126.604 336.916 131.868 337.11 137.086C337.676 152.284 335.641 167.235 333.401 182.2C331.815 192.802 330.878 203.502 329.278 214.101C328.417 219.807 327.28 225.578 325.321 230.976C323.759 235.279 321.196 239.409 318.317 243.006C311.585 251.42 303.104 257.68 292.893 261.414C288.381 263.064 283.952 265.016 279.332 266.275C273.076 267.98 266.711 269.338 260.33 270.509C250.605 272.292 240.844 273.878 231.07 275.381C220.672 276.98 210.306 277.306 199.939 274.719C194.33 273.32 188.527 272.723 182.869 271.504C166.828 268.049 151.043 263.651 135.754 257.669C130.918 255.776 126.25 253.478 122.199 249.956C118.49 246.731 113.928 244.469 110.316 241.155C103.357 234.766 96.6579 228.074 90.1249 221.245C84.3729 215.231 79.0449 208.814 73.4259 202.671C71.6229 200.7 69.3989 199.121 67.5219 197.212C61.8789 191.478 56.3579 185.624 50.6959 179.909C48.0139 177.202 45.0629 174.763 42.3439 172.091C39.7309 169.523 37.2799 166.791 34.7229 164.164C30.1899 159.507 25.8419 154.642 21.0319 150.288C14.4459 144.325 9.29194 137.288 4.85794 129.733C1.90494 124.702 0.404932 119.126 0.994932 113.109C1.35393 109.453 1.56894 105.873 3.02594 102.364C4.82294 98.043 7.59594 94.544 11.0199 91.581C16.4609 86.871 22.0179 82.28 27.7129 77.881C34.4159 72.703 41.2719 67.718 48.1519 62.774C53.0819 59.232 58.3649 56.157 63.1269 52.411C72.1059 45.348 81.2339 38.467 89.4079 30.405C96.0349 23.868 102.898 17.54 110.002 11.527C115.279 7.06004 121.135 3.23104 128.049 1.65704C130.639 1.07104 133.357 1.05302 136.018 0.775024ZM19.8459 102.8C15.5139 101.001 13.7579 101.522 12.1429 105.364C13.5239 105.867 14.8829 106.363 16.5709 106.978C16.7739 105.683 16.8949 104.912 16.9929 104.287C17.9989 103.763 18.9229 103.281 19.8479 102.799C21.2859 101.622 23.0749 100.717 23.4099 98.469C20.4119 98.883 20.4119 98.883 19.8459 102.8ZM118.352 15.815C117.153 17.925 116.342 19.402 117.231 21.328C119.746 19.487 119.773 19.382 118.352 15.815ZM36.2909 86.69C35.4119 88.799 34.8089 90.248 34.0939 91.961C37.8889 90.785 37.8889 90.785 36.2909 86.69ZM129.395 162.873C128.641 162.383 128.006 161.799 127.858 161.903C127.292 162.306 126.881 162.927 126.413 163.468C126.843 163.712 127.337 164.224 127.684 164.138C128.211 164.009 128.639 163.465 129.395 162.873ZM137.797 163.645C137.248 164.305 136.658 164.709 136.697 165.036C136.763 165.591 137.228 166.097 137.525 166.623C137.986 166.255 138.761 165.928 138.818 165.505C138.881 165.033 138.287 164.477 137.797 163.645ZM137.221 207.492C137.242 207.855 137.264 208.219 137.285 208.582C138.129 208.456 138.973 208.33 139.816 208.205C139.787 207.967 139.757 207.73 139.73 207.492C138.895 207.492 138.057 207.492 137.221 207.492ZM110.674 30.56C110.768 30.297 110.862 30.035 110.957 29.772C110.123 29.451 109.291 29.13 108.457 28.809C108.357 29.097 108.256 29.386 108.154 29.674C108.994 29.969 109.834 30.265 110.674 30.56ZM116.773 160.416C116.58 160.891 116.285 161.258 116.357 161.528C116.435 161.827 116.851 162.037 117.121 162.285C117.336 161.902 117.652 161.535 117.713 161.129C117.736 160.968 117.193 160.722 116.773 160.416ZM124.658 162.574C123.793 162.347 123.324 162.142 122.863 162.152C122.707 162.156 122.562 162.708 122.414 163.009C122.768 163.15 123.127 163.408 123.473 163.392C123.754 163.381 124.02 163.036 124.658 162.574ZM133.973 165.672C133.819 165.484 133.664 165.297 133.51 165.11C133.348 165.387 133.151 165.654 133.059 165.954C133.039 166.011 133.434 166.196 133.637 166.322C133.748 166.105 133.861 165.89 133.973 165.672ZM115.15 24.039C114.955 23.876 114.759 23.714 114.566 23.552C114.468 23.778 114.254 24.034 114.302 24.223C114.353 24.418 114.656 24.549 114.849 24.708C114.949 24.486 115.051 24.263 115.15 24.039Z"
				            fill="black" />
				          </svg>

				        </div>
				        <div class="decoration bottom left scale-2 blend-mode-multiply d-none d-md-block">
				          <svg class="bg-primary-2" width="257" height="200" viewBox="0 0 257 200" fill="none" xmlns="http://www.w3.org/2000/svg">
				            <path d="M52.4279 199.385C43.0839 199.209 36.3889 195.125 32.1679 186.768C27.0589 176.651 23.0699 166.059 18.5959 155.67C14.4909 146.139 11.7989 136.287 9.8339 126.129C8.3399 118.4 6.2499 110.777 4.2029 103.166C1.6089 93.512 1.1799 83.586 0.327904 73.727C0.0159039 70.11 0.526895 66.342 1.17789 62.735C3.21689 51.438 5.9829 40.346 11.6529 30.198C17.3889 19.928 24.6629 11.067 34.9749 5.09298C41.4319 1.35298 48.4239 -0.117996 55.8949 0.868004C57.9049 1.134 59.9499 1.2 61.9789 1.225C71.0429 1.332 79.9579 2.518 88.7739 4.635C99.0339 7.1 109.327 9.44598 119.891 10.244C124.323 10.578 128.77 10.765 133.196 11.16C142.811 12.017 152.415 13.006 162.03 13.869C169.618 14.551 177.214 15.588 184.811 15.637C194.612 15.698 203.692 17.942 212.225 22.553C219.284 26.367 225.918 30.826 232.161 35.899C237.325 40.096 239.942 45.866 242.524 51.674C245.827 59.108 248.79 66.699 252.278 74.043C253.758 77.166 254.821 80.272 255.723 83.602C258.006 92.018 256.088 99.834 252.858 107.36C247.544 119.743 238.922 129.653 228.809 138.475C223.024 143.522 217.065 148.209 210.334 151.858C201.387 156.712 192.336 161.282 182.279 163.602C175.226 165.229 168.332 167.526 161.322 169.358C150.377 172.221 139.363 174.829 128.441 177.782C124.039 178.973 119.779 180.723 115.502 182.346C108.85 184.867 102.33 187.789 95.5759 189.989C90.2869 191.712 84.7239 192.6 79.2829 193.852C73.0899 195.278 66.9079 196.752 60.6949 198.086C57.9729 198.67 55.1859 198.961 52.4279 199.385ZM2.4139 69.891C2.8909 77.291 2.8549 82.676 3.6699 87.928C4.8339 95.455 6.5329 102.903 8.1679 110.348C9.3969 115.936 11.0959 121.424 12.2069 127.03C13.8459 135.284 15.6869 143.446 19.1909 151.153C20.4529 153.928 21.7179 156.725 22.6929 159.61C25.8409 168.926 30.5069 177.551 34.7049 186.391C37.0959 191.42 40.8749 194.725 46.3479 195.953C47.9519 196.312 49.6189 196.803 51.2149 196.682C54.9999 196.397 58.7989 195.971 62.5239 195.246C66.6369 194.447 70.6199 192.838 74.7529 192.307C87.1079 190.719 98.8939 187.225 110.206 182.157C122.585 176.61 135.595 173.192 148.706 169.997C166.852 165.575 185.198 161.819 202.196 153.577C208.983 150.286 215.481 146.638 221.26 141.774C223.399 139.973 225.467 138.086 227.641 136.329C237.504 128.349 245.12 118.612 250.375 107.075C252.391 102.651 253.904 98.11 252.98 93.059C252.503 90.45 252.259 87.741 252.384 85.1C252.501 82.6 251.993 80.356 251.236 78.032C247.797 67.473 242.777 57.591 238.306 47.475C236.376 43.108 233.327 39.489 229.482 36.631C226.017 34.059 222.412 31.676 218.953 29.096C211.025 23.188 202.168 19.569 192.383 18.366C187.85 17.807 183.258 17.751 178.699 17.372C170.855 16.718 163.011 16.044 155.178 15.274C150.75 14.838 146.356 14.057 141.926 13.686C136.612 13.239 131.27 13.122 125.947 12.75C121.259 12.422 116.533 12.232 111.91 11.461C104.894 10.293 97.9309 8.775 90.9859 7.225C81.2789 5.059 71.5269 3.41799 61.5349 3.57699C59.8899 3.60399 58.2319 3.431 56.5989 3.208C47.8569 2.015 39.8019 4.02999 32.7179 9.07699C24.4229 14.985 18.0729 22.706 13.2099 31.694C9.53991 38.479 7.1459 45.712 5.4439 53.155C4.0329 59.334 3.1679 65.637 2.4139 69.891Z"
				            fill="black" />
				          </svg>

				        </div>
			      	</div>
			';
			
			if(!( 'none' == $settings['divider'] )){	
				echo tommusrhodus_svg_dividers_pluck( $settings['divider'], '' );		
			}
				
			echo '</section>';
			
		} elseif( 'dark-background-image-bottom' == $settings['layout'] ) {
			
			echo '
				<section class="bg-primary-3 text-light text-center has-divider header-desktop-app">
					<div class="container">
						<div class="row justify-content-center content-area">
							<div class="col-xl-7 col-lg-8 col-md-9">
								'. $settings['content'] .'
							</div>
						</div>';

						if( $settings['image']['id'] ) {


							echo '
							<div class="row justify-content-center mt-6" data-aos="fade-up" data-delay="100">
								<div class="col-lg-10">
									'. wp_get_attachment_image( $settings['image']['id'], 'full', 0, array( 'class' => 'rounded shadow-lg' ) ) .'
								</div>
							</div>
							';

						}

						echo '		
					</div>
			';
			
			if(!( 'none' == $settings['divider'] )){	
				echo tommusrhodus_svg_dividers_pluck( $settings['divider'], '' );		
			}
				
			echo '</section>';
			
		} elseif( 'centered-bg-primary-3-background' == $settings['layout'] ) {
			
			echo '
				<section class="bg-primary-3 has-divider text-light">
	
					<div class="container pb-6">
						<div class="row justify-content-center text-center">
							<div class="col-xl-8 col-lg-9 col-md-10">								
								'. $settings['content'] .'								
							</div>
						</div>
					</div>
			';
			
			if(!( 'none' == $settings['divider'] )){	
				echo tommusrhodus_svg_dividers_pluck( $settings['divider'], '' );		
			}
				
			echo '</section>';
			
		}
		
	}

}

// Register our new widget
Plugin::instance()->widgets_manager->register_widget_type( new Widget_TommusRhodus_Hero_Header_Block() );